#!/bin/bash

# 1. Nhập danh sách IP (Hỗ trợ nhập nhiều IP cách nhau bằng dấu cách hoặc dấu phẩy)
echo "-------------------------------------------------------"
echo "Ví dụ nhập IP Cluster: 172.0.0.3 172.0.0.4 172.0.0.5"
echo "-------------------------------------------------------"
read -p " IP Wazuh Manager Cluster (các node): " INPUT_IPS

# Chuyển đổi dấu phẩy thành dấu cách (đề phòng user nhập 10.0.0.1,10.0.0.2)
CLEAN_IPS=$(echo $INPUT_IPS | tr ',' ' ')

# Chuyển chuỗi thành mảng (Array)
IFS=' ' read -r -a WAZUH_MANAGERS <<< "$CLEAN_IPS"

# Kiểm tra input
if [ ${#WAZUH_MANAGERS[@]} -eq 0 ]; then
    echo "Error: Bạn chưa nhập IP nào cả."
    exit 1
fi

# Lấy IP đầu tiên để dùng cho việc Đăng ký (Enrollment) ban đầu
PRIMARY_MANAGER=${WAZUH_MANAGERS[0]}

WAZUH_GROUP="linux_agent"
WAZUH_AGENT_NAME=$(hostname)
AGENT_IP=$(hostname -I | awk '{print $1}')

echo "Installing Wazuh Agent for CLUSTER..."
echo "---------------------------------------------"
echo "Managers: ${WAZUH_MANAGERS[*]}"
echo "Primary (Auth): $PRIMARY_MANAGER"
echo "Group:    $WAZUH_GROUP"
echo "IP:       $AGENT_IP"
echo "---------------------------------------------"

if [[ $EUID -ne 0 ]]; then
   echo "Please run this script with root privileges (sudo)."
   exit 1
fi

# --- PHẦN CÀI ĐẶT (GIỮ NGUYÊN) ---
echo "Installing prerequisites..."
apt-get update
apt-get install -y curl gnupg lsb-release

echo "Adding Wazuh GPG key..."
curl -s https://packages.wazuh.com/key/GPG-KEY-WAZUH | gpg --no-default-keyring --keyring gnupg-ring:/usr/share/keyrings/wazuh.gpg --import && chmod 644 /usr/share/keyrings/wazuh.gpg

echo "Adding Wazuh repository..."
echo "deb [signed-by=/usr/share/keyrings/wazuh.gpg] https://packages.wazuh.com/4.x/apt/ stable main" | tee /etc/apt/sources.list.d/wazuh.list

echo "Installing Wazuh Agent package..."
apt-get update
apt-get install wazuh-agent -y

# --- PHẦN ĐĂNG KÝ (ENROLLMENT) ---
# Chỉ cần đăng ký với 1 node (thường là Master hoặc node bất kỳ trong cluster)
echo "Registering the agent with Primary Manager ($PRIMARY_MANAGER)..."
/var/ossec/bin/agent-auth -m "$PRIMARY_MANAGER" -A "$WAZUH_AGENT_NAME" -G "$WAZUH_GROUP" -I "$AGENT_IP"


# --- PHẦN CẤU HÌNH CLUSTER (QUAN TRỌNG) ---
echo "Configuring ossec.conf for Cluster..."

# 1. Xóa block <server> mặc định cũ đi (để tránh trùng lặp)
# Lệnh sed này xóa từ dòng chứa <server> đến dòng chứa </server> đầu tiên tìm thấy
sed -i '/<server>/,/<\/server>/d' /var/ossec/etc/ossec.conf

# 2. Tạo chuỗi XML mới chứa nhiều server
NEW_SERVER_BLOCK=""
for ip in "${WAZUH_MANAGERS[@]}"; do
    NEW_SERVER_BLOCK+="\
    <server>\n\
      <address>$ip</address>\n\
      <port>1514</port>\n\
      <protocol>tcp</protocol>\n\
    </server>\n"
done

# 3. Chèn chuỗi XML mới vào ngay sau thẻ <client>
# Dùng Perl để xử lý thay thế chuỗi nhiều dòng dễ hơn sed
perl -i -pe "s|<client>|<client>\n$NEW_SERVER_BLOCK|" /var/ossec/etc/ossec.conf

# --- KHỞI ĐỘNG ---
echo "Starting service..."
systemctl daemon-reload
systemctl enable wazuh-agent
systemctl restart wazuh-agent

echo "---------------------------------------------"
echo "DONE! Agent configured with Cluster nodes:"
for ip in "${WAZUH_MANAGERS[@]}"; do
   echo " - Node: $ip"
done
echo "---------------------------------------------"